---
name: mnexium-app-builder
description: Build and harden Mnexium-powered applications with concrete code changes. Use when users ask to scaffold integrations, implement chat/memory/profiles/state/claims/records features, design endpoint mappings, add retries/observability/tests, or debug Mnexium API usage in JavaScript/TypeScript/Python. Trigger on requests like "build an app with Mnexium", "wire mnx subject_id/chat_id", "add memory recall", "implement records CRUD", "productionize Mnexium integration", or when user invokes $mnexium-app-builder. Do not use for unrelated generic coding tasks with no Mnexium API/component.
---

# Mnexium App Builder

Use this skill when the user is building an app on Mnexium and needs implementation help, not just API explanation.
Mnexium integration patterns are framework-agnostic: support web, mobile, backend, CLI, or agent workflows in any stack.

## Trigger Tuning

Should trigger:
- "Build a chat app using Mnexium memory."
- "Add `mnx.subject_id` and stable `chat_id` handling."
- "Implement Mnexium Records schema and CRUD endpoints."
- "Harden Mnexium integration with retries and metrics."

Should not trigger:
- generic React UI styling without Mnexium integration
- unrelated SQL/schema tasks not using Mnexium endpoints
- non-technical content writing requests

## Inputs To Collect

Collect only what is needed to start:
- runtime/language (for example `JavaScript/TypeScript`, `Python`, `Go`, `Ruby`, `Java`, `C#`)
- app surface (web app, mobile app, backend API, CLI, agent worker)
- model/provider target: `OpenAI`, `Anthropic`, `Google`, or mixed
- identity source for `mnx.subject_id`
- conversation model for `mnx.chat_id` (per thread/session)
- feature scope: chat only, or memory/profiles/state/claims/records

If details are missing, choose safe defaults and continue.

## Execution Contract (Required)

1. Prefer implementation over explanation. Make file edits and run checks unless user explicitly asks for design-only guidance.
2. For existing repos, inspect local route handlers/types first and follow local contracts exactly.
3. Treat `llms.txt` as fallback when local code does not define a contract.
4. Return concrete artifacts:
   - changed files
   - commands run and outcomes
   - unresolved blockers (if any)
5. Avoid broad API essays when the user asked for a fix.

## Default Build Rules

1. Resolve contract source in this order:
   - local code in current workspace
   - `https://www.mnexium.com/llms.txt`
   - user-provided payload examples
2. Use `x-mnexium-key` for Mnexium auth unless user requests otherwise.
3. Keep provider and Mnexium keys server-side.
4. Enforce stable `subject_id` and `chat_id` across turns.
5. Add retries with backoff for `429` and `5xx`.
6. Normalize Mnexium/provider errors into app-level error objects.
7. Add integration tests for chat continuity and feature-specific behavior.
8. Never invent Records endpoints; do not call `/api/v1/records/events` unless the contract explicitly adds it.
9. Never invent Memory Policy endpoints; use only `/api/v1/memory/policies`, `/api/v1/memory/policies/:id`, and `/api/v1/memory/policies/resolve`.

## High-Risk Contract Traps (Always Check)

- Records schema create may require `type_name` (not `type`).
- Records create/update usually require `data` object (not `record`).
- `/api/v1/records/:type` infers type from URL; schema route may still require `type_name`.
- `mnx.records.sync=true` is typically non-stream only and may return `422` when no writes happen.
- If request trace is missing, verify the request path is one that actually logs audit rows.

## Canonical mnx Object

Use this baseline in chat-style requests and adjust feature flags based on product behavior:

```json
{
  "mnx": {
    "subject_id": "user_123",
    "chat_id": "chat_456",
    "history": true,
    "learn": true,
    "recall": true,
    "summarize": "balanced",
    "memory_policy": "mem_pol_abc123"
  }
}
```

Rules:
- `subject_id` must be stable per authenticated user.
- `chat_id` must be stable per conversation thread.
- enable/disable `history`, `learn`, `recall`, and `summarize` per use case.
- `memory_policy` supports: omitted/`true` (default scoped resolution), `false` (disable policy), or explicit policy ID.

## Memory Policy Endpoints

- `GET /api/v1/memory/policies`
- `POST /api/v1/memory/policies`
- `GET /api/v1/memory/policies/:id`
- `PATCH /api/v1/memory/policies/:id`
- `DELETE /api/v1/memory/policies/:id`
- `GET /api/v1/memory/policies/resolve`

## Implementation Workflow

1. Map user journey to Mnexium endpoints.
2. Implement a thin Mnexium adapter service/module.
3. Add app-level input validation before outbound API calls.
4. Add observability: latency, failures, memory recall usage.
5. Add tests and run them.
6. Return changed files plus any setup/env requirements.

When working inside an existing repository:
1. Locate relevant API route handlers and shared helpers.
2. Reproduce error with current request/response shape.
3. Patch minimal files to fix contract mismatch.
4. Run build/tests/lint that exist in repo.
5. Summarize exact before/after payload differences.

## Resource Selection

Use only the minimum needed references:
- For endpoint coverage, read `references/endpoint-map.md`.
- For environment variable setup, read `references/env-example.md`.
- For production guardrails and acceptance checks, read `references/production-checklist.md`.
- For trigger validation and regression checks, read `references/trigger-tests.md`.
- For failure recovery patterns, read `references/troubleshooting.md`.

## Output Requirements

When shipping changes:
- include concrete file edits
- preserve existing project conventions
- call out assumptions
- include verification results or say what could not be run

## Required QA Pass

Before finalizing, run the trigger set in `references/trigger-tests.md`:
- confirm expected trigger behavior on Mnexium tasks
- confirm non-trigger behavior on unrelated tasks
- update this skill if over-triggering or under-triggering is observed
